/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.core.io.buffer;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.Channel;
import java.nio.channels.Channels;
import java.nio.channels.CompletionHandler;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscription;
import org.springframework.core.io.Resource;
import org.springframework.core.io.buffer.CloseableDataBuffer;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferFactory;
import org.springframework.core.io.buffer.LimitedDataBufferList;
import org.springframework.core.io.buffer.OutputStreamPublisher;
import org.springframework.core.io.buffer.PooledDataBuffer;
import org.springframework.core.io.buffer.TouchableDataBuffer;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import reactor.core.CoreSubscriber;
import reactor.core.Disposable;
import reactor.core.publisher.BaseSubscriber;
import reactor.core.publisher.Flux;
import reactor.core.publisher.FluxSink;
import reactor.core.publisher.Mono;
import reactor.core.publisher.MonoSink;
import reactor.core.publisher.SynchronousSink;
import reactor.util.context.Context;
import reactor.util.context.ContextView;

public abstract class DataBufferUtils {
    private static final Log logger = LogFactory.getLog(DataBufferUtils.class);
    private static final Consumer<DataBuffer> RELEASE_CONSUMER = DataBufferUtils::release;
    private static final int DEFAULT_CHUNK_SIZE = 1024;

    public static Flux<DataBuffer> readInputStream(Callable<InputStream> inputStreamSupplier, DataBufferFactory bufferFactory, int bufferSize) {
        Assert.notNull(inputStreamSupplier, "'inputStreamSupplier' must not be null");
        return DataBufferUtils.readByteChannel(() -> Channels.newChannel((InputStream)inputStreamSupplier.call()), bufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> readByteChannel(Callable<ReadableByteChannel> channelSupplier, DataBufferFactory bufferFactory, int bufferSize) {
        Assert.notNull(channelSupplier, "'channelSupplier' must not be null");
        Assert.notNull((Object)bufferFactory, "'bufferFactory' must not be null");
        Assert.isTrue(bufferSize > 0, "'bufferSize' must be > 0");
        return Flux.using(channelSupplier, channel -> Flux.generate((Consumer)new ReadableByteChannelGenerator((ReadableByteChannel)channel, bufferFactory, bufferSize)), DataBufferUtils::closeChannel);
    }

    public static Flux<DataBuffer> readAsynchronousFileChannel(Callable<AsynchronousFileChannel> channelSupplier, DataBufferFactory bufferFactory, int bufferSize) {
        return DataBufferUtils.readAsynchronousFileChannel(channelSupplier, 0L, bufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> readAsynchronousFileChannel(Callable<AsynchronousFileChannel> channelSupplier, long position, DataBufferFactory bufferFactory, int bufferSize) {
        Assert.notNull(channelSupplier, "'channelSupplier' must not be null");
        Assert.notNull((Object)bufferFactory, "'bufferFactory' must not be null");
        Assert.isTrue(position >= 0L, "'position' must be >= 0");
        Assert.isTrue(bufferSize > 0, "'bufferSize' must be > 0");
        Flux flux = Flux.using(channelSupplier, channel -> Flux.create(sink -> {
            ReadCompletionHandler handler = new ReadCompletionHandler((AsynchronousFileChannel)channel, (FluxSink<DataBuffer>)sink, position, bufferFactory, bufferSize);
            sink.onCancel(handler::cancel);
            sink.onRequest(handler::request);
        }), channel -> {});
        return flux.doOnDiscard(DataBuffer.class, DataBufferUtils::release);
    }

    public static Flux<DataBuffer> read(Path path2, DataBufferFactory bufferFactory, int bufferSize, OpenOption ... options2) {
        Assert.notNull((Object)path2, "Path must not be null");
        Assert.notNull((Object)bufferFactory, "DataBufferFactory must not be null");
        Assert.isTrue(bufferSize > 0, "'bufferSize' must be > 0");
        if (options2.length > 0) {
            for (OpenOption option : options2) {
                Assert.isTrue(option != StandardOpenOption.APPEND && option != StandardOpenOption.WRITE, () -> "'" + option + "' not allowed");
            }
        }
        return DataBufferUtils.readAsynchronousFileChannel(() -> AsynchronousFileChannel.open(path2, options2), bufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> read(Resource resource, DataBufferFactory bufferFactory, int bufferSize) {
        return DataBufferUtils.read(resource, 0L, bufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> read(Resource resource, long position, DataBufferFactory bufferFactory, int bufferSize) {
        try {
            if (resource.isFile()) {
                File file2 = resource.getFile();
                return DataBufferUtils.readAsynchronousFileChannel(() -> AsynchronousFileChannel.open(file2.toPath(), StandardOpenOption.READ), position, bufferFactory, bufferSize);
            }
        }
        catch (IOException file2) {
        }
        Flux<DataBuffer> result2 = DataBufferUtils.readByteChannel(resource::readableChannel, bufferFactory, bufferSize);
        return position == 0L ? result2 : DataBufferUtils.skipUntilByteCount(result2, position);
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source2, OutputStream outputStream) {
        Assert.notNull(source2, "'source' must not be null");
        Assert.notNull((Object)outputStream, "'outputStream' must not be null");
        WritableByteChannel channel = Channels.newChannel(outputStream);
        return DataBufferUtils.write(source2, channel);
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source2, WritableByteChannel channel) {
        Assert.notNull(source2, "'source' must not be null");
        Assert.notNull((Object)channel, "'channel' must not be null");
        Flux flux = Flux.from(source2);
        return Flux.create(sink -> {
            WritableByteChannelSubscriber subscriber = new WritableByteChannelSubscriber((FluxSink<DataBuffer>)sink, channel);
            sink.onDispose((Disposable)subscriber);
            flux.subscribe((CoreSubscriber)subscriber);
        });
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source2, AsynchronousFileChannel channel) {
        return DataBufferUtils.write(source2, channel, 0L);
    }

    public static Flux<DataBuffer> write(Publisher<? extends DataBuffer> source2, AsynchronousFileChannel channel, long position) {
        Assert.notNull(source2, "'source' must not be null");
        Assert.notNull((Object)channel, "'channel' must not be null");
        Assert.isTrue(position >= 0L, "'position' must be >= 0");
        Flux flux = Flux.from(source2);
        return Flux.create(sink -> {
            WriteCompletionHandler handler = new WriteCompletionHandler((FluxSink<DataBuffer>)sink, channel, position);
            sink.onDispose((Disposable)handler);
            flux.subscribe((CoreSubscriber)handler);
        });
    }

    public static Mono<Void> write(Publisher<DataBuffer> source2, Path destination, OpenOption ... options2) {
        Assert.notNull(source2, "Source must not be null");
        Assert.notNull((Object)destination, "Destination must not be null");
        Set<OpenOption> optionSet = DataBufferUtils.checkWriteOptions(options2);
        return Mono.create(sink -> {
            try {
                AsynchronousFileChannel channel = AsynchronousFileChannel.open(destination, optionSet, null, new FileAttribute[0]);
                sink.onDispose(() -> DataBufferUtils.closeChannel(channel));
                DataBufferUtils.write(source2, channel).subscribe(DataBufferUtils::release, arg_0 -> ((MonoSink)sink).error(arg_0), () -> ((MonoSink)sink).success(), Context.of((ContextView)sink.contextView()));
            }
            catch (IOException ex) {
                sink.error((Throwable)ex);
            }
        });
    }

    private static Set<OpenOption> checkWriteOptions(OpenOption[] options2) {
        int length2 = options2.length;
        HashSet<OpenOption> result2 = new HashSet<OpenOption>(length2 + 3);
        if (length2 == 0) {
            result2.add(StandardOpenOption.CREATE);
            result2.add(StandardOpenOption.TRUNCATE_EXISTING);
        } else {
            for (OpenOption opt : options2) {
                if (opt == StandardOpenOption.READ) {
                    throw new IllegalArgumentException("READ not allowed");
                }
                result2.add(opt);
            }
        }
        result2.add(StandardOpenOption.WRITE);
        return result2;
    }

    static void closeChannel(@Nullable Channel channel) {
        if (channel != null && channel.isOpen()) {
            try {
                channel.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public static Publisher<DataBuffer> outputStreamPublisher(Consumer<OutputStream> outputStreamConsumer, DataBufferFactory bufferFactory, Executor executor) {
        return DataBufferUtils.outputStreamPublisher(outputStreamConsumer, bufferFactory, executor, 1024);
    }

    public static Publisher<DataBuffer> outputStreamPublisher(Consumer<OutputStream> outputStreamConsumer, DataBufferFactory bufferFactory, Executor executor, int chunkSize) {
        Assert.notNull(outputStreamConsumer, "OutputStreamConsumer must not be null");
        Assert.notNull((Object)bufferFactory, "BufferFactory must not be null");
        Assert.notNull((Object)executor, "Executor must not be null");
        Assert.isTrue(chunkSize > 0, "Chunk size must be > 0");
        return new OutputStreamPublisher(outputStreamConsumer, bufferFactory, executor, chunkSize);
    }

    public static <T extends DataBuffer> Flux<T> takeUntilByteCount(Publisher<T> publisher, long maxByteCount) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(maxByteCount >= 0L, "'maxByteCount' must be >= 0");
        return Flux.defer(() -> {
            AtomicLong countDown = new AtomicLong(maxByteCount);
            return Flux.from((Publisher)publisher).map(buffer -> {
                long remainder2 = countDown.addAndGet(-buffer.readableByteCount());
                if (remainder2 < 0L) {
                    int index2 = buffer.readableByteCount() + (int)remainder2;
                    DataBuffer split2 = buffer.split(index2);
                    DataBufferUtils.release(buffer);
                    return split2;
                }
                return buffer;
            }).takeUntil(buffer -> countDown.get() <= 0L);
        });
    }

    public static <T extends DataBuffer> Flux<T> skipUntilByteCount(Publisher<T> publisher, long maxByteCount) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(maxByteCount >= 0L, "'maxByteCount' must be >= 0");
        return Flux.defer(() -> {
            AtomicLong countDown = new AtomicLong(maxByteCount);
            return Flux.from((Publisher)publisher).skipUntil(buffer -> {
                long remainder2 = countDown.addAndGet(-buffer.readableByteCount());
                return remainder2 < 0L;
            }).map(buffer -> {
                long remainder2 = countDown.get();
                if (remainder2 < 0L) {
                    countDown.set(0L);
                    int start2 = buffer.readableByteCount() + (int)remainder2;
                    DataBuffer split2 = buffer.split(start2);
                    DataBufferUtils.release(split2);
                    return buffer;
                }
                return buffer;
            });
        }).doOnDiscard(DataBuffer.class, DataBufferUtils::release);
    }

    public static <T extends DataBuffer> T retain(T dataBuffer) {
        if (dataBuffer instanceof PooledDataBuffer) {
            PooledDataBuffer pooledDataBuffer = (PooledDataBuffer)dataBuffer;
            return (T)pooledDataBuffer.retain();
        }
        return dataBuffer;
    }

    public static <T extends DataBuffer> T touch(T dataBuffer, Object hint) {
        if (dataBuffer instanceof TouchableDataBuffer) {
            TouchableDataBuffer touchableDataBuffer = (TouchableDataBuffer)dataBuffer;
            return (T)touchableDataBuffer.touch(hint);
        }
        return dataBuffer;
    }

    public static boolean release(@Nullable DataBuffer dataBuffer) {
        if (dataBuffer instanceof PooledDataBuffer) {
            PooledDataBuffer pooledDataBuffer = (PooledDataBuffer)dataBuffer;
            if (pooledDataBuffer.isAllocated()) {
                try {
                    return pooledDataBuffer.release();
                }
                catch (IllegalStateException ex) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Failed to release PooledDataBuffer: " + dataBuffer, ex);
                    }
                    return false;
                }
            }
        } else if (dataBuffer instanceof CloseableDataBuffer) {
            CloseableDataBuffer closeableDataBuffer = (CloseableDataBuffer)dataBuffer;
            try {
                closeableDataBuffer.close();
                return true;
            }
            catch (IllegalStateException ex) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Failed to release CloseableDataBuffer " + dataBuffer, ex);
                }
                return false;
            }
        }
        return false;
    }

    public static Consumer<DataBuffer> releaseConsumer() {
        return RELEASE_CONSUMER;
    }

    public static Mono<DataBuffer> join(Publisher<? extends DataBuffer> dataBuffers) {
        return DataBufferUtils.join(dataBuffers, -1);
    }

    public static Mono<DataBuffer> join(Publisher<? extends DataBuffer> buffers, int maxByteCount) {
        Assert.notNull(buffers, "'buffers' must not be null");
        if (buffers instanceof Mono) {
            Mono mono = (Mono)buffers;
            return mono;
        }
        return Flux.from(buffers).collect(() -> new LimitedDataBufferList(maxByteCount), LimitedDataBufferList::add).filter(list2 -> !list2.isEmpty()).map(list2 -> ((DataBuffer)list2.get(0)).factory().join((List<? extends DataBuffer>)list2)).doOnDiscard(DataBuffer.class, DataBufferUtils::release);
    }

    public static Matcher matcher(byte[] delimiter) {
        return DataBufferUtils.createMatcher(delimiter);
    }

    public static Matcher matcher(byte[] ... delimiters) {
        Assert.isTrue(delimiters.length > 0, "Delimiters must not be empty");
        return delimiters.length == 1 ? DataBufferUtils.createMatcher(delimiters[0]) : new CompositeMatcher(delimiters);
    }

    private static NestedMatcher createMatcher(byte[] delimiter) {
        int length2 = delimiter.length;
        Assert.isTrue(length2 > 0, "Delimiter must not be empty");
        return switch (length2) {
            case 1 -> {
                if (delimiter[0] == 10) {
                    yield SingleByteMatcher.NEWLINE_MATCHER;
                }
                yield new SingleByteMatcher(delimiter);
            }
            case 2 -> new TwoByteMatcher(delimiter);
            default -> new KnuthMorrisPrattMatcher(delimiter);
        };
    }

    private static interface NestedMatcher
    extends Matcher {
        public boolean match(byte var1);
    }

    private static class CompositeMatcher
    implements Matcher {
        private static final byte[] NO_DELIMITER = new byte[0];
        private final NestedMatcher[] matchers;
        byte[] longestDelimiter = NO_DELIMITER;

        CompositeMatcher(byte[][] delimiters) {
            this.matchers = CompositeMatcher.initMatchers(delimiters);
        }

        private static NestedMatcher[] initMatchers(byte[][] delimiters) {
            NestedMatcher[] matchers = new NestedMatcher[delimiters.length];
            for (int i2 = 0; i2 < delimiters.length; ++i2) {
                matchers[i2] = DataBufferUtils.createMatcher(delimiters[i2]);
            }
            return matchers;
        }

        @Override
        public int match(DataBuffer dataBuffer) {
            this.longestDelimiter = NO_DELIMITER;
            for (int pos2 = dataBuffer.readPosition(); pos2 < dataBuffer.writePosition(); ++pos2) {
                byte b2 = dataBuffer.getByte(pos2);
                for (NestedMatcher matcher : this.matchers) {
                    if (!matcher.match(b2) || matcher.delimiter().length <= this.longestDelimiter.length) continue;
                    this.longestDelimiter = matcher.delimiter();
                }
                if (this.longestDelimiter == NO_DELIMITER) continue;
                this.reset();
                return pos2;
            }
            return -1;
        }

        @Override
        public byte[] delimiter() {
            Assert.state(this.longestDelimiter != NO_DELIMITER, "'delimiter' not set");
            return this.longestDelimiter;
        }

        @Override
        public void reset() {
            for (NestedMatcher matcher : this.matchers) {
                matcher.reset();
            }
        }
    }

    private static class SingleByteMatcher
    implements NestedMatcher {
        static final SingleByteMatcher NEWLINE_MATCHER = new SingleByteMatcher(new byte[]{10});
        private final byte[] delimiter;

        SingleByteMatcher(byte[] delimiter) {
            Assert.isTrue(delimiter.length == 1, "Expected a 1 byte delimiter");
            this.delimiter = delimiter;
        }

        @Override
        public int match(DataBuffer dataBuffer) {
            for (int pos2 = dataBuffer.readPosition(); pos2 < dataBuffer.writePosition(); ++pos2) {
                byte b2 = dataBuffer.getByte(pos2);
                if (!this.match(b2)) continue;
                return pos2;
            }
            return -1;
        }

        @Override
        public boolean match(byte b2) {
            return this.delimiter[0] == b2;
        }

        @Override
        public byte[] delimiter() {
            return this.delimiter;
        }

        @Override
        public void reset() {
        }
    }

    private static class TwoByteMatcher
    extends AbstractNestedMatcher {
        protected TwoByteMatcher(byte[] delimiter) {
            super(delimiter);
            Assert.isTrue(delimiter.length == 2, "Expected a 2-byte delimiter");
        }
    }

    private static class KnuthMorrisPrattMatcher
    extends AbstractNestedMatcher {
        private final int[] table;

        public KnuthMorrisPrattMatcher(byte[] delimiter) {
            super(delimiter);
            this.table = KnuthMorrisPrattMatcher.longestSuffixPrefixTable(delimiter);
        }

        private static int[] longestSuffixPrefixTable(byte[] delimiter) {
            int[] result2 = new int[delimiter.length];
            result2[0] = 0;
            for (int i2 = 1; i2 < delimiter.length; ++i2) {
                int j = result2[i2 - 1];
                while (j > 0 && delimiter[i2] != delimiter[j]) {
                    j = result2[j - 1];
                }
                if (delimiter[i2] == delimiter[j]) {
                    // empty if block
                }
                result2[i2] = ++j;
            }
            return result2;
        }

        @Override
        public boolean match(byte b2) {
            while (this.getMatches() > 0 && b2 != this.delimiter()[this.getMatches()]) {
                this.setMatches(this.table[this.getMatches() - 1]);
            }
            return super.match(b2);
        }
    }

    private static class WriteCompletionHandler
    extends BaseSubscriber<DataBuffer>
    implements CompletionHandler<Integer, Attachment> {
        private final FluxSink<DataBuffer> sink;
        private final AsynchronousFileChannel channel;
        private final AtomicBoolean writing = new AtomicBoolean();
        private final AtomicBoolean completed = new AtomicBoolean();
        private final AtomicReference<Throwable> error = new AtomicReference();
        private final AtomicLong position;

        public WriteCompletionHandler(FluxSink<DataBuffer> sink, AsynchronousFileChannel channel, long position) {
            this.sink = sink;
            this.channel = channel;
            this.position = new AtomicLong(position);
        }

        protected void hookOnSubscribe(Subscription subscription) {
            this.request(1L);
        }

        protected void hookOnNext(DataBuffer dataBuffer) {
            DataBuffer.ByteBufferIterator iterator = dataBuffer.readableByteBuffers();
            if (iterator.hasNext()) {
                ByteBuffer byteBuffer = (ByteBuffer)iterator.next();
                long pos2 = this.position.get();
                Attachment attachment = new Attachment(byteBuffer, dataBuffer, iterator);
                this.writing.set(true);
                this.channel.write(byteBuffer, pos2, attachment, this);
            }
        }

        protected void hookOnError(Throwable throwable) {
            this.error.set(throwable);
            if (!this.writing.get()) {
                this.sink.error(throwable);
            }
        }

        protected void hookOnComplete() {
            this.completed.set(true);
            if (!this.writing.get()) {
                this.sink.complete();
            }
        }

        @Override
        public void completed(Integer written, Attachment attachment) {
            DataBuffer.ByteBufferIterator iterator = attachment.iterator();
            iterator.close();
            long pos2 = this.position.addAndGet(written.intValue());
            ByteBuffer byteBuffer = attachment.byteBuffer();
            if (byteBuffer.hasRemaining()) {
                this.channel.write(byteBuffer, pos2, attachment, this);
            } else if (iterator.hasNext()) {
                ByteBuffer next2 = (ByteBuffer)iterator.next();
                this.channel.write(next2, pos2, attachment, this);
            } else {
                this.sink.next((Object)attachment.dataBuffer());
                this.writing.set(false);
                Throwable throwable = this.error.get();
                if (throwable != null) {
                    this.sink.error(throwable);
                } else if (this.completed.get()) {
                    this.sink.complete();
                } else {
                    this.request(1L);
                }
            }
        }

        @Override
        public void failed(Throwable ex, Attachment attachment) {
            attachment.iterator().close();
            this.sink.next((Object)attachment.dataBuffer());
            this.writing.set(false);
            this.sink.error(ex);
        }

        public Context currentContext() {
            return Context.of((ContextView)this.sink.contextView());
        }

        private record Attachment(ByteBuffer byteBuffer, DataBuffer dataBuffer, DataBuffer.ByteBufferIterator iterator) {
        }
    }

    private static class WritableByteChannelSubscriber
    extends BaseSubscriber<DataBuffer> {
        private final FluxSink<DataBuffer> sink;
        private final WritableByteChannel channel;

        public WritableByteChannelSubscriber(FluxSink<DataBuffer> sink, WritableByteChannel channel) {
            this.sink = sink;
            this.channel = channel;
        }

        protected void hookOnSubscribe(Subscription subscription) {
            this.request(1L);
        }

        protected void hookOnNext(DataBuffer dataBuffer) {
            try {
                try (DataBuffer.ByteBufferIterator iterator = dataBuffer.readableByteBuffers();){
                    ByteBuffer byteBuffer = (ByteBuffer)iterator.next();
                    while (byteBuffer.hasRemaining()) {
                        this.channel.write(byteBuffer);
                    }
                }
                this.sink.next((Object)dataBuffer);
                this.request(1L);
            }
            catch (IOException ex) {
                this.sink.next((Object)dataBuffer);
                this.sink.error((Throwable)ex);
            }
        }

        protected void hookOnError(Throwable throwable) {
            this.sink.error(throwable);
        }

        protected void hookOnComplete() {
            this.sink.complete();
        }

        public Context currentContext() {
            return Context.of((ContextView)this.sink.contextView());
        }
    }

    private static class ReadCompletionHandler
    implements CompletionHandler<Integer, Attachment> {
        private final AsynchronousFileChannel channel;
        private final FluxSink<DataBuffer> sink;
        private final DataBufferFactory dataBufferFactory;
        private final int bufferSize;
        private final AtomicLong position;
        private final AtomicReference<State> state = new AtomicReference<State>(State.IDLE);

        public ReadCompletionHandler(AsynchronousFileChannel channel, FluxSink<DataBuffer> sink, long position, DataBufferFactory dataBufferFactory, int bufferSize) {
            this.channel = channel;
            this.sink = sink;
            this.position = new AtomicLong(position);
            this.dataBufferFactory = dataBufferFactory;
            this.bufferSize = bufferSize;
        }

        public void request(long n) {
            this.tryRead();
        }

        public void cancel() {
            this.state.getAndSet(State.DISPOSED);
            DataBufferUtils.closeChannel(this.channel);
        }

        private void tryRead() {
            if (this.sink.requestedFromDownstream() > 0L && this.state.compareAndSet(State.IDLE, State.READING)) {
                this.read();
            }
        }

        private void read() {
            DataBuffer dataBuffer = this.dataBufferFactory.allocateBuffer(this.bufferSize);
            DataBuffer.ByteBufferIterator iterator = dataBuffer.writableByteBuffers();
            Assert.state(iterator.hasNext(), "No ByteBuffer available");
            ByteBuffer byteBuffer = (ByteBuffer)iterator.next();
            Attachment attachment = new Attachment(dataBuffer, iterator);
            this.channel.read(byteBuffer, this.position.get(), attachment, this);
        }

        @Override
        public void completed(Integer read2, Attachment attachment) {
            attachment.iterator().close();
            DataBuffer dataBuffer = attachment.dataBuffer();
            if (this.state.get() == State.DISPOSED) {
                DataBufferUtils.release(dataBuffer);
                DataBufferUtils.closeChannel(this.channel);
                return;
            }
            if (read2 == -1) {
                DataBufferUtils.release(dataBuffer);
                DataBufferUtils.closeChannel(this.channel);
                this.state.set(State.DISPOSED);
                this.sink.complete();
                return;
            }
            this.position.addAndGet(read2.intValue());
            dataBuffer.writePosition(read2);
            this.sink.next((Object)dataBuffer);
            if (this.sink.requestedFromDownstream() > 0L) {
                this.read();
                return;
            }
            if (this.state.compareAndSet(State.READING, State.IDLE)) {
                this.tryRead();
            }
        }

        @Override
        public void failed(Throwable ex, Attachment attachment) {
            attachment.iterator().close();
            DataBufferUtils.release(attachment.dataBuffer());
            DataBufferUtils.closeChannel(this.channel);
            this.state.set(State.DISPOSED);
            this.sink.error(ex);
        }

        private static enum State {
            IDLE,
            READING,
            DISPOSED;

        }

        private record Attachment(DataBuffer dataBuffer, DataBuffer.ByteBufferIterator iterator) {
        }
    }

    private static class ReadableByteChannelGenerator
    implements Consumer<SynchronousSink<DataBuffer>> {
        private final ReadableByteChannel channel;
        private final DataBufferFactory dataBufferFactory;
        private final int bufferSize;

        public ReadableByteChannelGenerator(ReadableByteChannel channel, DataBufferFactory dataBufferFactory, int bufferSize) {
            this.channel = channel;
            this.dataBufferFactory = dataBufferFactory;
            this.bufferSize = bufferSize;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void accept(SynchronousSink<DataBuffer> sink) {
            int read2 = -1;
            DataBuffer dataBuffer = this.dataBufferFactory.allocateBuffer(this.bufferSize);
            try {
                try (DataBuffer.ByteBufferIterator iterator = dataBuffer.writableByteBuffers();){
                    Assert.state(iterator.hasNext(), "No ByteBuffer available");
                    ByteBuffer byteBuffer = (ByteBuffer)iterator.next();
                    read2 = this.channel.read(byteBuffer);
                }
                if (read2 >= 0) {
                    dataBuffer.writePosition(read2);
                    sink.next((Object)dataBuffer);
                } else {
                    sink.complete();
                }
            }
            catch (IOException ex) {
                sink.error((Throwable)ex);
            }
            finally {
                if (read2 == -1) {
                    DataBufferUtils.release(dataBuffer);
                }
            }
        }
    }

    private static abstract class AbstractNestedMatcher
    implements NestedMatcher {
        private final byte[] delimiter;
        private int matches = 0;

        protected AbstractNestedMatcher(byte[] delimiter) {
            this.delimiter = delimiter;
        }

        protected void setMatches(int index2) {
            this.matches = index2;
        }

        protected int getMatches() {
            return this.matches;
        }

        @Override
        public int match(DataBuffer dataBuffer) {
            for (int pos2 = dataBuffer.readPosition(); pos2 < dataBuffer.writePosition(); ++pos2) {
                byte b2 = dataBuffer.getByte(pos2);
                if (!this.match(b2)) continue;
                this.reset();
                return pos2;
            }
            return -1;
        }

        @Override
        public boolean match(byte b2) {
            if (b2 == this.delimiter[this.matches]) {
                ++this.matches;
                return this.matches == this.delimiter().length;
            }
            return false;
        }

        @Override
        public byte[] delimiter() {
            return this.delimiter;
        }

        @Override
        public void reset() {
            this.matches = 0;
        }
    }

    public static interface Matcher {
        public int match(DataBuffer var1);

        public byte[] delimiter();

        public void reset();
    }
}

