/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.ssl.pem;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.AlgorithmParameters;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.HexFormat;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.Cipher;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.springframework.util.Assert;

final class PemPrivateKeyParser {
    private static final String PKCS1_RSA_HEADER = "-+BEGIN\\s+RSA\\s+PRIVATE\\s+KEY[^-]*-+(?:\\s|\\r|\\n)+";
    private static final String PKCS1_RSA_FOOTER = "-+END\\s+RSA\\s+PRIVATE\\s+KEY[^-]*-+";
    private static final String PKCS8_HEADER = "-+BEGIN\\s+PRIVATE\\s+KEY[^-]*-+(?:\\s|\\r|\\n)+";
    private static final String PKCS8_FOOTER = "-+END\\s+PRIVATE\\s+KEY[^-]*-+";
    private static final String PKCS8_ENCRYPTED_HEADER = "-+BEGIN\\s+ENCRYPTED\\s+PRIVATE\\s+KEY[^-]*-+(?:\\s|\\r|\\n)+";
    private static final String PKCS8_ENCRYPTED_FOOTER = "-+END\\s+ENCRYPTED\\s+PRIVATE\\s+KEY[^-]*-+";
    private static final String SEC1_EC_HEADER = "-+BEGIN\\s+EC\\s+PRIVATE\\s+KEY[^-]*-+(?:\\s|\\r|\\n)+";
    private static final String SEC1_EC_FOOTER = "-+END\\s+EC\\s+PRIVATE\\s+KEY[^-]*-+";
    private static final String BASE64_TEXT = "([a-z0-9+/=\\r\\n]+)";
    public static final int BASE64_TEXT_GROUP = 1;
    private static final EncodedOid RSA_ALGORITHM = EncodedOid.OID_1_2_840_113549_1_1_1;
    private static final EncodedOid ELLIPTIC_CURVE_ALGORITHM = EncodedOid.OID_1_2_840_10045_2_1;
    private static final EncodedOid ELLIPTIC_CURVE_384_BIT = EncodedOid.OID_1_3_132_0_34;
    private static final Map<EncodedOid, String> ALGORITHMS;
    private static final List<PemParser> PEM_PARSERS;

    private PemPrivateKeyParser() {
    }

    private static PKCS8EncodedKeySpec createKeySpecForPkcs1Rsa(byte[] bytes2, String password) {
        return PemPrivateKeyParser.createKeySpecForAlgorithm(bytes2, RSA_ALGORITHM, null);
    }

    private static PKCS8EncodedKeySpec createKeySpecForSec1Ec(byte[] bytes2, String password) {
        DerElement ecPrivateKey = DerElement.of(bytes2);
        Assert.state(ecPrivateKey.isType(DerElement.ValueType.ENCODED, DerElement.TagType.SEQUENCE), "Key spec should be an ASN.1 encoded sequence");
        DerElement version = DerElement.of(ecPrivateKey.getContents());
        Assert.state(version != null && version.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.INTEGER), "Key spec should start with version");
        Assert.state(version.getContents().remaining() == 1 && version.getContents().get() == 1, "Key spec version must be 1");
        DerElement privateKey = DerElement.of(ecPrivateKey.getContents());
        Assert.state(privateKey != null && privateKey.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.OCTET_STRING), "Key spec should contain private key");
        DerElement parameters2 = DerElement.of(ecPrivateKey.getContents());
        return PemPrivateKeyParser.createKeySpecForAlgorithm(bytes2, ELLIPTIC_CURVE_ALGORITHM, PemPrivateKeyParser.getEcParameters(parameters2));
    }

    private static EncodedOid getEcParameters(DerElement parameters2) {
        if (parameters2 == null) {
            return ELLIPTIC_CURVE_384_BIT;
        }
        Assert.state(parameters2.isType(DerElement.ValueType.ENCODED), "Key spec should contain encoded parameters");
        DerElement contents = DerElement.of(parameters2.getContents());
        Assert.state(contents != null && contents.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.OBJECT_IDENTIFIER), "Key spec parameters should contain object identifier");
        return EncodedOid.of(contents);
    }

    private static PKCS8EncodedKeySpec createKeySpecForAlgorithm(byte[] bytes2, EncodedOid algorithm, EncodedOid parameters2) {
        try {
            DerEncoder encoder = new DerEncoder();
            encoder.integer(0);
            DerEncoder algorithmIdentifier = new DerEncoder();
            algorithmIdentifier.objectIdentifier(algorithm);
            algorithmIdentifier.objectIdentifier(parameters2);
            encoder.sequence(algorithmIdentifier.toByteArray());
            encoder.octetString(bytes2);
            return new PKCS8EncodedKeySpec(encoder.toSequence());
        }
        catch (IOException ex) {
            throw new IllegalStateException(ex);
        }
    }

    private static PKCS8EncodedKeySpec createKeySpecForPkcs8(byte[] bytes2, String password) {
        DerElement ecPrivateKey = DerElement.of(bytes2);
        Assert.state(ecPrivateKey.isType(DerElement.ValueType.ENCODED, DerElement.TagType.SEQUENCE), "Key spec should be an ASN.1 encoded sequence");
        DerElement version = DerElement.of(ecPrivateKey.getContents());
        Assert.state(version != null && version.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.INTEGER), "Key spec should start with version");
        DerElement sequence = DerElement.of(ecPrivateKey.getContents());
        Assert.state(sequence != null && sequence.isType(DerElement.ValueType.ENCODED, DerElement.TagType.SEQUENCE), "Key spec should contain private key");
        DerElement algorithmId = DerElement.of(sequence.getContents());
        Assert.state(algorithmId != null && algorithmId.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.OBJECT_IDENTIFIER), "Key spec container object identifier");
        String algorithmName = ALGORITHMS.get(EncodedOid.of(algorithmId));
        return algorithmName != null ? new PKCS8EncodedKeySpec(bytes2, algorithmName) : new PKCS8EncodedKeySpec(bytes2);
    }

    private static PKCS8EncodedKeySpec createKeySpecForPkcs8Encrypted(byte[] bytes2, String password) {
        return Pkcs8PrivateKeyDecryptor.decrypt(bytes2, password);
    }

    static PrivateKey parse(String text) {
        return PemPrivateKeyParser.parse(text, null);
    }

    static PrivateKey parse(String text, String password) {
        if (text == null) {
            return null;
        }
        try {
            for (PemParser pemParser : PEM_PARSERS) {
                PrivateKey privateKey = pemParser.parse(text, password);
                if (privateKey == null) continue;
                return privateKey;
            }
        }
        catch (Exception ex) {
            throw new IllegalStateException("Error loading private key file: " + ex.getMessage(), ex);
        }
        throw new IllegalStateException("Missing private key or unrecognized format");
    }

    static {
        HashMap<EncodedOid, String> algorithms = new HashMap<EncodedOid, String>();
        algorithms.put(EncodedOid.OID_1_2_840_113549_1_1_1, "RSA");
        algorithms.put(EncodedOid.OID_1_2_840_113549_1_1_10, "RSA");
        algorithms.put(EncodedOid.OID_1_2_840_10040_4_1, "DSA");
        algorithms.put(EncodedOid.OID_1_3_101_110, "XDH");
        algorithms.put(EncodedOid.OID_1_3_101_111, "XDH");
        algorithms.put(EncodedOid.OID_1_3_101_112, "EdDSA");
        algorithms.put(EncodedOid.OID_1_3_101_113, "EdDSA");
        algorithms.put(EncodedOid.OID_1_2_840_10045_2_1, "EC");
        ALGORITHMS = Collections.unmodifiableMap(algorithms);
        ArrayList<PemParser> parsers = new ArrayList<PemParser>();
        parsers.add(new PemParser(PKCS1_RSA_HEADER, PKCS1_RSA_FOOTER, PemPrivateKeyParser::createKeySpecForPkcs1Rsa, "RSA"));
        parsers.add(new PemParser(SEC1_EC_HEADER, SEC1_EC_FOOTER, PemPrivateKeyParser::createKeySpecForSec1Ec, "EC"));
        parsers.add(new PemParser(PKCS8_HEADER, PKCS8_FOOTER, PemPrivateKeyParser::createKeySpecForPkcs8, "RSA", "RSASSA-PSS", "EC", "DSA", "EdDSA", "XDH"));
        parsers.add(new PemParser(PKCS8_ENCRYPTED_HEADER, PKCS8_ENCRYPTED_FOOTER, PemPrivateKeyParser::createKeySpecForPkcs8Encrypted, "RSA", "RSASSA-PSS", "EC", "DSA", "EdDSA", "XDH"));
        PEM_PARSERS = Collections.unmodifiableList(parsers);
    }

    static final class EncodedOid {
        static final EncodedOid OID_1_2_840_10040_4_1 = EncodedOid.of("2a8648ce380401");
        static final EncodedOid OID_1_2_840_113549_1_1_1 = EncodedOid.of("2A864886F70D010101");
        static final EncodedOid OID_1_2_840_113549_1_1_10 = EncodedOid.of("2a864886f70d01010a");
        static final EncodedOid OID_1_3_101_110 = EncodedOid.of("2b656e");
        static final EncodedOid OID_1_3_101_111 = EncodedOid.of("2b656f");
        static final EncodedOid OID_1_3_101_112 = EncodedOid.of("2b6570");
        static final EncodedOid OID_1_3_101_113 = EncodedOid.of("2b6571");
        static final EncodedOid OID_1_2_840_10045_2_1 = EncodedOid.of("2a8648ce3d0201");
        static final EncodedOid OID_1_3_132_0_34 = EncodedOid.of("2b81040022");
        private final byte[] value;

        private EncodedOid(byte[] value2) {
            this.value = value2;
        }

        byte[] toByteArray() {
            return (byte[])this.value.clone();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            return Arrays.equals(this.value, ((EncodedOid)obj).value);
        }

        public int hashCode() {
            return Arrays.hashCode(this.value);
        }

        static EncodedOid of(String hexString) {
            return EncodedOid.of(HexFormat.of().parseHex(hexString));
        }

        static EncodedOid of(DerElement derElement) {
            return EncodedOid.of(derElement.getContents());
        }

        static EncodedOid of(ByteBuffer byteBuffer) {
            return EncodedOid.of(byteBuffer.array(), byteBuffer.arrayOffset() + byteBuffer.position(), byteBuffer.remaining());
        }

        static EncodedOid of(byte[] bytes2) {
            return EncodedOid.of(bytes2, 0, bytes2.length);
        }

        static EncodedOid of(byte[] bytes2, int off, int len) {
            byte[] value2 = new byte[len];
            System.arraycopy(bytes2, off, value2, 0, len);
            return new EncodedOid(value2);
        }
    }

    static final class DerElement {
        private final ValueType valueType;
        private final long tagType;
        private final ByteBuffer contents;

        private DerElement(ByteBuffer bytes2) {
            byte b2 = bytes2.get();
            this.valueType = (b2 & 0x20) == 0 ? ValueType.PRIMITIVE : ValueType.ENCODED;
            this.tagType = this.decodeTagType(b2, bytes2);
            int length2 = this.decodeLength(bytes2);
            bytes2.limit(bytes2.position() + length2);
            this.contents = bytes2.slice();
            bytes2.limit(bytes2.capacity());
            bytes2.position(bytes2.position() + length2);
        }

        private long decodeTagType(byte b2, ByteBuffer bytes2) {
            long tagType = b2 & 0x1F;
            if (tagType != 31L) {
                return tagType;
            }
            tagType = 0L;
            b2 = bytes2.get();
            while ((b2 & 0x80) != 0) {
                tagType <<= 7;
                tagType |= (long)(b2 & 0x7F);
                b2 = bytes2.get();
            }
            return tagType;
        }

        private int decodeLength(ByteBuffer bytes2) {
            byte b2 = bytes2.get();
            if ((b2 & 0x80) == 0) {
                return b2 & 0x7F;
            }
            int numberOfLengthBytes = b2 & 0x7F;
            Assert.state(numberOfLengthBytes != 0, "Infinite length encoding is not supported");
            Assert.state(numberOfLengthBytes != 127, "Reserved length encoding is not supported");
            Assert.state(numberOfLengthBytes <= 4, "Length overflow");
            int length2 = 0;
            for (int i2 = 0; i2 < numberOfLengthBytes; ++i2) {
                length2 <<= 8;
                length2 |= bytes2.get() & 0xFF;
            }
            return length2;
        }

        boolean isType(ValueType valueType) {
            return this.valueType == valueType;
        }

        boolean isType(ValueType valueType, TagType tagType) {
            return this.valueType == valueType && this.tagType == (long)tagType.getNumber();
        }

        ByteBuffer getContents() {
            return this.contents;
        }

        static DerElement of(byte[] bytes2) {
            return DerElement.of(ByteBuffer.wrap(bytes2));
        }

        static DerElement of(ByteBuffer bytes2) {
            return bytes2.remaining() > 0 ? new DerElement(bytes2) : null;
        }

        static enum ValueType {
            PRIMITIVE,
            ENCODED;

        }

        static enum TagType {
            INTEGER(2),
            OCTET_STRING(4),
            OBJECT_IDENTIFIER(6),
            SEQUENCE(16);

            private final int number;

            private TagType(int number) {
                this.number = number;
            }

            int getNumber() {
                return this.number;
            }
        }
    }

    static class DerEncoder {
        private final ByteArrayOutputStream stream = new ByteArrayOutputStream();

        DerEncoder() {
        }

        void objectIdentifier(EncodedOid encodedOid) throws IOException {
            int code = encodedOid != null ? 6 : 5;
            this.codeLengthBytes(code, encodedOid != null ? encodedOid.toByteArray() : null);
        }

        void integer(int ... encodedInteger) throws IOException {
            this.codeLengthBytes(2, DerEncoder.bytes(encodedInteger));
        }

        void octetString(byte[] bytes2) throws IOException {
            this.codeLengthBytes(4, bytes2);
        }

        void sequence(byte[] bytes2) throws IOException {
            this.codeLengthBytes(48, bytes2);
        }

        void codeLengthBytes(int code, byte[] bytes2) throws IOException {
            int length2;
            this.stream.write(code);
            int n = length2 = bytes2 != null ? bytes2.length : 0;
            if (length2 <= 127) {
                this.stream.write(length2 & 0xFF);
            } else {
                ByteArrayOutputStream lengthStream = new ByteArrayOutputStream();
                while (length2 != 0) {
                    lengthStream.write(length2 & 0xFF);
                    length2 >>= 8;
                }
                byte[] lengthBytes = lengthStream.toByteArray();
                this.stream.write(0x80 | lengthBytes.length);
                for (int i2 = lengthBytes.length - 1; i2 >= 0; --i2) {
                    this.stream.write(lengthBytes[i2]);
                }
            }
            if (bytes2 != null) {
                this.stream.write(bytes2);
            }
        }

        private static byte[] bytes(int ... elements) {
            if (elements == null) {
                return null;
            }
            byte[] result2 = new byte[elements.length];
            for (int i2 = 0; i2 < elements.length; ++i2) {
                result2[i2] = (byte)elements[i2];
            }
            return result2;
        }

        byte[] toSequence() throws IOException {
            DerEncoder sequenceEncoder = new DerEncoder();
            sequenceEncoder.sequence(this.toByteArray());
            return sequenceEncoder.toByteArray();
        }

        byte[] toByteArray() {
            return this.stream.toByteArray();
        }
    }

    static class Pkcs8PrivateKeyDecryptor {
        public static final String PBES2_ALGORITHM = "PBES2";

        Pkcs8PrivateKeyDecryptor() {
        }

        static PKCS8EncodedKeySpec decrypt(byte[] bytes2, String password) {
            Assert.notNull((Object)password, "Password is required for an encrypted private key");
            try {
                EncryptedPrivateKeyInfo keyInfo = new EncryptedPrivateKeyInfo(bytes2);
                AlgorithmParameters algorithmParameters = keyInfo.getAlgParameters();
                String encryptionAlgorithm = Pkcs8PrivateKeyDecryptor.getEncryptionAlgorithm(algorithmParameters, keyInfo.getAlgName());
                SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(encryptionAlgorithm);
                SecretKey key2 = keyFactory.generateSecret(new PBEKeySpec(password.toCharArray()));
                Cipher cipher = Cipher.getInstance(encryptionAlgorithm);
                cipher.init(2, (Key)key2, algorithmParameters);
                return keyInfo.getKeySpec(cipher);
            }
            catch (IOException | GeneralSecurityException ex) {
                throw new IllegalArgumentException("Error decrypting private key", ex);
            }
        }

        private static String getEncryptionAlgorithm(AlgorithmParameters algParameters, String algName) {
            if (algParameters != null && PBES2_ALGORITHM.equals(algName)) {
                return algParameters.toString();
            }
            return algName;
        }
    }

    private static class PemParser {
        private final Pattern pattern;
        private final BiFunction<byte[], String, PKCS8EncodedKeySpec> keySpecFactory;
        private final String[] algorithms;

        PemParser(String header, String footer, BiFunction<byte[], String, PKCS8EncodedKeySpec> keySpecFactory, String ... algorithms) {
            this.pattern = Pattern.compile(header + PemPrivateKeyParser.BASE64_TEXT + footer, 2);
            this.keySpecFactory = keySpecFactory;
            this.algorithms = algorithms;
        }

        PrivateKey parse(String text, String password) {
            Matcher matcher = this.pattern.matcher(text);
            return !matcher.find() ? null : this.parse(PemParser.decodeBase64(matcher.group(1)), password);
        }

        private static byte[] decodeBase64(String content) {
            byte[] contentBytes = content.replaceAll("\r", "").replaceAll("\n", "").getBytes();
            return Base64.getDecoder().decode(contentBytes);
        }

        private PrivateKey parse(byte[] bytes2, String password) {
            PKCS8EncodedKeySpec keySpec = this.keySpecFactory.apply(bytes2, password);
            if (keySpec.getAlgorithm() != null) {
                try {
                    KeyFactory keyFactory = KeyFactory.getInstance(keySpec.getAlgorithm());
                    return keyFactory.generatePrivate(keySpec);
                }
                catch (NoSuchAlgorithmException | InvalidKeySpecException generalSecurityException) {
                    // empty catch block
                }
            }
            for (String algorithm : this.algorithms) {
                try {
                    KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
                    return keyFactory.generatePrivate(keySpec);
                }
                catch (NoSuchAlgorithmException | InvalidKeySpecException generalSecurityException) {
                }
            }
            return null;
        }
    }
}

