/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.autoconfigure.ssl;

import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.time.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.core.log.LogMessage;
import org.springframework.util.Assert;

class FileWatcher
implements Closeable {
    private static final Log logger = LogFactory.getLog(FileWatcher.class);
    private final Duration quietPeriod;
    private final Object lock = new Object();
    private WatcherThread thread;

    FileWatcher(Duration quietPeriod) {
        Assert.notNull((Object)quietPeriod, "QuietPeriod must not be null");
        this.quietPeriod = quietPeriod;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void watch(Set<Path> paths, Runnable action) {
        Assert.notNull(paths, "Paths must not be null");
        Assert.notNull((Object)action, "Action must not be null");
        if (paths.isEmpty()) {
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            try {
                if (this.thread == null) {
                    this.thread = new WatcherThread();
                    this.thread.start();
                }
                this.thread.register(new Registration(paths, action));
            }
            catch (IOException ex) {
                throw new UncheckedIOException("Failed to register paths for watching: " + paths, ex);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        Object object = this.lock;
        synchronized (object) {
            if (this.thread != null) {
                this.thread.close();
                this.thread.interrupt();
                try {
                    this.thread.join();
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
                this.thread = null;
            }
        }
    }

    private class WatcherThread
    extends Thread
    implements Closeable {
        private final WatchService watchService = FileSystems.getDefault().newWatchService();
        private final Map<WatchKey, List<Registration>> registrations = new ConcurrentHashMap<WatchKey, List<Registration>>();
        private volatile boolean running = true;

        WatcherThread() throws IOException {
            this.setName("ssl-bundle-watcher");
            this.setDaemon(true);
            this.setUncaughtExceptionHandler(this::onThreadException);
        }

        private void onThreadException(Thread thread2, Throwable throwable) {
            logger.error("Uncaught exception in file watcher thread", throwable);
        }

        void register(Registration registration) throws IOException {
            for (Path path2 : registration.paths()) {
                if (!Files.isRegularFile(path2, new LinkOption[0]) && !Files.isDirectory(path2, new LinkOption[0])) {
                    throw new IOException("'%s' is neither a file nor a directory".formatted(path2));
                }
                Path directory = Files.isDirectory(path2, new LinkOption[0]) ? path2 : path2.getParent();
                WatchKey watchKey = this.register(directory);
                this.registrations.computeIfAbsent(watchKey, key2 -> new CopyOnWriteArrayList()).add(registration);
            }
        }

        private WatchKey register(Path directory) throws IOException {
            logger.debug(LogMessage.format("Registering '%s'", (Object)directory));
            return directory.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_DELETE);
        }

        @Override
        public void run() {
            logger.debug("Watch thread started");
            HashSet<Runnable> actions = new HashSet<Runnable>();
            while (this.running) {
                try {
                    long timeout2 = FileWatcher.this.quietPeriod.toMillis();
                    WatchKey key2 = this.watchService.poll(timeout2, TimeUnit.MILLISECONDS);
                    if (key2 == null) {
                        actions.forEach(this::runSafely);
                        actions.clear();
                        continue;
                    }
                    this.accumulate(key2, actions);
                    key2.reset();
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
                catch (ClosedWatchServiceException ex) {
                    logger.debug("File watcher has been closed");
                    this.running = false;
                }
            }
            logger.debug("Watch thread stopped");
        }

        private void runSafely(Runnable action) {
            try {
                action.run();
            }
            catch (Throwable ex) {
                logger.error("Unexpected SSL reload error", ex);
            }
        }

        private void accumulate(WatchKey key2, Set<Runnable> actions) {
            List<Registration> registrations = this.registrations.get(key2);
            Path directory = (Path)key2.watchable();
            for (WatchEvent<?> event2 : key2.pollEvents()) {
                Path file2 = directory.resolve((Path)event2.context());
                for (Registration registration : registrations) {
                    if (!registration.manages(file2)) continue;
                    actions.add(registration.action());
                }
            }
        }

        @Override
        public void close() throws IOException {
            this.running = false;
            this.watchService.close();
        }
    }

    private record Registration(Set<Path> paths, Runnable action) {
        Registration {
            paths = paths.stream().map(Path::toAbsolutePath).collect(Collectors.toSet());
        }

        boolean manages(Path file2) {
            Path absolutePath = file2.toAbsolutePath();
            return this.paths.contains(absolutePath) || this.isInDirectories(absolutePath);
        }

        private boolean isInDirectories(Path file2) {
            return this.paths.stream().filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).anyMatch(file2::startsWith);
        }
    }
}

